﻿using System.Collections.Concurrent;
using System.Net.WebSockets;
using System.Security.Cryptography;
using System.Text;
using System.Text.Json;
using ObsStudioTest.Logger;

namespace ObsStudioTest.ObsStudio;

public sealed class ObsWebSocketClient(ObsOptions opt, ILoggerAsync loggerAsync) : IObsClient
{
    public bool IsConnected
    {
        get
        {
            bool isConnected = _ws is { State: WebSocketState.Open } && _isConnected;
            _ = loggerAsync.Log($"isConnected: {isConnected}");
            return isConnected;
        }
    }

    private volatile bool _isConnected;
    private ClientWebSocket? _ws;
    private Task? _receiveLoop;
    private readonly ConcurrentDictionary<string, TaskCompletionSource<JsonElement>> _pending = new();
    private readonly SemaphoreSlim _sendLock = new(1, 1);
    private readonly SemaphoreSlim _connectLock = new(1, 1);

    public async Task<bool> ConnectAsync(CancellationToken ct = default)
    {
        if (IsConnected) return true;

        await _connectLock.WaitAsync(ct);
        try
        {
            if (IsConnected) return true;

            await DisposeSocketAsync();

            _ws = new ClientWebSocket();
            _ws.Options.AddSubProtocol("obswebsocket.json");
            await _ws.ConnectAsync(opt.Uri, ct);

            // Hello (op=0)
            var hello = await ReceiveJsonAsync(_ws, ct);
            if (hello.GetProperty("op").GetInt32() != 0)
                throw new Exception("Expected OBS Hello (op=0).");

            var helloD = hello.GetProperty("d");
            int rpcVersion = helloD.GetProperty("rpcVersion").GetInt32();

            string? auth = null;
            if (helloD.TryGetProperty("authentication", out var authObj))
            {
                var challenge = authObj.GetProperty("challenge").GetString()!;
                var salt = authObj.GetProperty("salt").GetString()!;
                auth = CreateObsAuthString(opt.Password, salt, challenge);
            }

            // Identify (op=1)
            var identify = new
            {
                op = 1,
                d = new
                {
                    rpcVersion,
                    authentication = auth,
                    eventSubscriptions = opt.EventSubscriptions
                }
            };
            await SendJsonAsync(_ws, identify, ct);

            var identified = await ReceiveJsonAsync(_ws, ct);
            if (identified.GetProperty("op").GetInt32() != 2)
                throw new Exception("OBS Identify failed (expected op=2).");

            _isConnected = true;

            var ws = _ws;
            _receiveLoop = Task.Run(() => ReceiveLoopAsync(ws, CancellationToken.None));

            await loggerAsync.Log($"OBS connected: {opt.Uri}");
            return true;
        }
        catch (OperationCanceledException)
        {
            await loggerAsync.Log("OBS connect canceled.");
            _isConnected = false;
            await DisposeSocketAsync();
            return false;
        }
        catch (Exception ex)
        {
            _isConnected = false;
            await loggerAsync.Log(ex);

            await DisposeSocketAsync();
            return false;
        }
        finally
        {
            _connectLock.Release();
        }
    }


    public async Task<bool> SendAsync(string requestType, object? requestData = null, CancellationToken ct = default)
    {
        await loggerAsync.Log($"requestType: {requestType}");
        if (!await ConnectAsync(ct))
        {
            return false;
        }

        var ws = _ws;
        if (ws is null || ws.State != WebSocketState.Open || !_isConnected)
            return false;

        var requestId = Guid.NewGuid().ToString("N");
        var tcs = new TaskCompletionSource<JsonElement>(TaskCreationOptions.RunContinuationsAsynchronously);
        _pending[requestId] = tcs;

        try
        {
            var payload = new
            {
                op = 6,
                d = new { requestType, requestId, requestData = requestData ?? new { } }
            };

            await _sendLock.WaitAsync(ct);
            try
            {
                await SendJsonAsync(ws, payload, ct);
            }
            finally
            {
                _sendLock.Release();
            }

            using var timeoutCts = CancellationTokenSource.CreateLinkedTokenSource(ct);
            timeoutCts.CancelAfter(TimeSpan.FromSeconds(15));

            var msg = await tcs.Task.WaitAsync(timeoutCts.Token);

            var d = msg.GetProperty("d");
            var status = d.GetProperty("requestStatus");
            if (!status.GetProperty("result").GetBoolean())
            {
                var code = status.TryGetProperty("code", out var c) ? c.GetInt32() : -1;
                var comment = status.TryGetProperty("comment", out var cm) ? cm.GetString() : "(no comment)";
                throw new Exception($"OBS request '{requestType}' failed. code={code}, comment={comment}");
            }

            return true;
        }
        catch (Exception ex)
        {
            await loggerAsync.Log(ex);
            return false;
        }
        finally
        {
            _pending.TryRemove(requestId, out _);
        }
    }

    private async Task ReceiveLoopAsync(ClientWebSocket ws, CancellationToken ct)
    {
        try
        {
            while (ws.State == WebSocketState.Open && !ct.IsCancellationRequested)
            {
                var msg = await ReceiveJsonAsync(ws, ct);
                var op = msg.GetProperty("op").GetInt32();

                if (op == 7)
                {
                    var d = msg.GetProperty("d");
                    var requestId = d.GetProperty("requestId").GetString();
                    if (requestId != null && _pending.TryGetValue(requestId, out var tcs))
                        tcs.TrySetResult(msg);
                }
            }
        }
        catch (Exception ex)
        {
            _isConnected = false;
            await loggerAsync.Log(ex);

            var pendingSnapshot = _pending.ToArray();
            foreach (var kv in pendingSnapshot)
            {
                kv.Value.TrySetException(ex);
                _pending.TryRemove(kv.Key, out _);
            }
        }
    }

    private static async Task SendJsonAsync(ClientWebSocket ws, object payload, CancellationToken ct)
    {
        var json = JsonSerializer.Serialize(payload, new JsonSerializerOptions
        {
            PropertyNamingPolicy = null,
            DefaultIgnoreCondition = System.Text.Json.Serialization.JsonIgnoreCondition.WhenWritingNull
        });

        var bytes = Encoding.UTF8.GetBytes(json);
        await ws.SendAsync(bytes, WebSocketMessageType.Text, endOfMessage: true, ct);
    }

    private static async Task<JsonElement> ReceiveJsonAsync(ClientWebSocket ws, CancellationToken ct)
    {
        var buffer = new byte[64 * 1024];
        using var ms = new MemoryStream();

        while (true)
        {
            var result = await ws.ReceiveAsync(buffer, ct);
            if (result.MessageType == WebSocketMessageType.Close)
                throw new Exception($"OBS WebSocket closed: {ws.CloseStatus} {ws.CloseStatusDescription}");

            ms.Write(buffer, 0, result.Count);
            if (result.EndOfMessage) break;
        }

        using var doc = JsonDocument.Parse(ms.ToArray());
        return doc.RootElement.Clone();
    }

    private static string CreateObsAuthString(string? password, string salt, string challenge)
    {
        if (string.IsNullOrWhiteSpace(password))
            throw new Exception("OBS requires authentication but password is empty.");

        var sha = SHA256.HashData(Encoding.UTF8.GetBytes(password + salt));
        var base64Secret = Convert.ToBase64String(sha);

        var sha2 = SHA256.HashData(Encoding.UTF8.GetBytes(base64Secret + challenge));
        return Convert.ToBase64String(sha2);
    }

    public async ValueTask DisposeAsync()
    {
        await DisposeSocketAsync();
        _sendLock.Dispose();
        _connectLock.Dispose();
    }

    private async Task DisposeSocketAsync()
    {
        if (_ws == null) return;

        _isConnected = false;

        try
        {
            _ws.Abort(); // wichtig, stoppt Receive sofort

            if (_ws.State == WebSocketState.Open)
                await _ws.CloseAsync(WebSocketCloseStatus.NormalClosure, "Closing", CancellationToken.None);
        }
        catch { /* ignore */ }

        _ws.Dispose();
        _ws = null;
    }
}